<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Blog;
use Illuminate\Support\Str;

class BlogController extends Controller
{
    public function index()
    {
        $blogs = Blog::latest()->paginate(10);
        return view('admin.blog.index', compact('blogs'));
    }

    public function create()
    {
        return view('admin.blog.addedit');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|unique:blogs,title',
            'body' => 'required',
            'image' => 'nullable|image|mimes:jpg,jpeg,png|dimensions:width=1169,height=414',
            'video' => 'nullable|string'
        ], [
            'image.dimensions' => 'The image must be exactly 1169x414 pixels.',
        ]);

        // Move image to /public/assets/admin/blog-images
        $imagePath = null;
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = time().'_'.$file->getClientOriginalName();
            $file->move(public_path('assets/admin/blog-images'), $filename);
            $imagePath = 'assets/admin/blog-images/' . $filename;
        }

        Blog::create([
            'title' => $request->title,
            'body' => $request->body,
            'image' => $imagePath,
            'video' => $request->video,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
        ]);

        return redirect()->route('admin.blogs.index')->with('success', 'Blog created successfully.');
    }

    public function edit(Blog $blog)
    {
        return view('admin.blog.addedit', compact('blog'));
    }

    public function update(Request $request, Blog $blog)
    {
        $request->validate([
            'title' => 'required|unique:blogs,title,' . $blog->id,
            'body' => 'required',
            'image' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'video' => 'nullable|string'
        ]);

        // Keep old image path unless a new image is uploaded
        $imagePath = $blog->image;

        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = time().'_'.$file->getClientOriginalName();
            $file->move(public_path('assets/admin/blog-images'), $filename);
            $imagePath = 'assets/admin/blog-images/' . $filename;
        }

        $blog->update([
            'title' => $request->title,
            'body' => $request->body,
            'image' => $imagePath,
            'video' => $request->video,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
        ]);

        return redirect()->route('admin.blogs.index')->with('success', 'Blog updated successfully.');
    }


    public function destroy(Blog $blog)
    {
        $blog->delete();
        return back()->with('success', 'Blog deleted.');
    }
}
